/************************************************************************************************
 * Twilight : this class contains the SUN physics : the passive igloo project
 * Twilight 2016 is Copyright (c) 2016, Peter Gallinelli
 * 
 * created : February 2016 by peter gallinelli
 * history : 
 ************************************************************************************************/

import static java.lang.Math.PI;
import static java.lang.Math.sin;
import static java.lang.Math.cos;
import static java.lang.Math.tan;
import static java.lang.Math.acos;
import static java.lang.Math.asin;
import static java.lang.Math.atan2;
import static java.lang.Math.exp;
import static java.lang.Math.pow;
import static java.lang.Math.floor;
import static java.lang.Math.abs;
import static java.lang.Math.toRadians;
import static java.lang.Math.toDegrees;

public class sunCalc {
    
    int[] JM = new int[14];  // cumulated day per month [doy]

    void setJM(int[] J) {
        for (int i = 0; i < J.length; i++)
            JM[i] = J[i];
    }

    /**
     * Day of year : transforms month and day to day of year (doy)
     * @param month int: month [1...12]
     * @param day int: day [1...31]
     * @return int: day of year [1...365]
     */
    public int dateToDOY(int month, int day) {
        int n = JM[month];
        return n + day;
    }

    /**
     * Date : transforms hour of year into month, day, hour aund minute
     * @param hoy double : hour of year [1...(yearDays * 24)]
     * @return double[] : [0] month, [1] day, [2] hour, [3] minute
     */
    public double[] HOYtoDate(double hoy) {
        int doy = (int) floor(hoy / 24);
        int hh = (int) hoy - doy * 24;
        int min = (int) ((hoy - floor(hoy)) * 60);
        int mm = 1;
        while (doy > JM[mm])
            mm++;
        int dd = doy - JM[mm - 1];
        double[] result = { mm, dd, hh, min };
        return result;
    }

    /**
     * Calculate obstruction angle from a list of summits
     * @param az double : azimuth to evaluate [deg]
     * @param horizonData double[][] : table containing horizon data = list of summits [cf. twilight]
     * @return double : horizon altitude angle for azimuth [deg]
     */
    public double getHorizonAngle(double az, double[][] horizonData) {
        // search for apropriate azimut interval
        int i = 1;
        while (i < horizonData.length && az > horizonData[0][i]) {
            i++;
        }
        // calculate linear interpolation between points
        double m = (horizonData[1][i] - horizonData[1][i - 1]) / (horizonData[0][i] - horizonData[0][i - 1]);
        double b = horizonData[1][i] - m * horizonData[0][i];
        double horHc = m * az + b;
        return horHc;
    }
    
    /**
     * Transposition : calculates solar radiation on tilted surface for date (year, month, day)
     * @param lat double : latitude [deg]
     * @param lon double : longitude [deg]
     * @param alt double : altitude [m]
     * @param inc double : inclinaison tilted surface [deg; 0 = horizontal; 90 = vertical]
     * @param paz double : azimut of tilted surface [deg; 0 = north; 90 = east; 180 = south; ...]
     * @param year double : time
     * @param month
     * @param day
     * @param zone double : timezone
     * @param dls double : dylight saving ?
     * @return Gs double: daily global solar radiation [kWh/m2]
     */
    public double getTransPositionDoy(double lat, double lon, double alt, double paz, double inc, double albedo,
                                      double year, double month, double day, double zone, double dls, double[][] horizonData) {
        double SUM_Gs = 0;
        double min = 0;
        double sec = 0;
        for (double hour = 0; hour < 24; hour++) {
            SUM_Gs += getTransposition(lat, lon, alt, paz, inc, albedo, year, month, day, hour, min, sec, zone, dls, horizonData);
        }
        return SUM_Gs;
    }
    
    /**
     * Override method for getTransposition : calculates without obstructions (set to zero)
     * @param lat double
     * @param lon double
     * @param alt double
     * @param paz double
     * @param inc double
     * @param albedo double
     * @param year double
     * @param month double
     * @param day double
     * @param zone double
     * @param dls double
     * @return double : daily global solar radiation [kWh/m2]
     */
    public double getTranspositionDoy(double lat, double lon, double alt, double paz, double inc, double albedo,
                        double year, double month, double day, double zone, double dls) {
        double[][] horizonData = {{ 0, 180, 360}, {0, 0, 0}}; // horizon dummy
        double Gs = getTransPositionDoy(lat, lon, alt, paz, inc, albedo, year, month, day, zone, dls, horizonData);
        return Gs;
    }
    
    /**
     * Calculate suns declination (overrrides calcSunDeclination(t) method)
     * @param year double
     * @param month double
     * @param day double
     * @param hour double
     * @param min double
     * @param sec double
     * @param zone double : time zone
     * @param dls double : dayligh savings [0] = off [1] = on
     * @return double : suns declination [deg]
     */
    public double calcSunDeclination(double year, double month, double day, double hour, double min, double sec, double zone, double dls) {
        // change time zone to positive hours in western hemisphere
        double daySavings = dls * 60;
        double hh = hour - (daySavings / 60);

        // timenow is GMT time for calculation in hours since 0Z
        double timenow = hh + min / 60 + sec / 3600 + zone * -1;
        double JD = calcJD(year, month, day);
        double t = calcTimeJulianCent(JD + timenow / 24);
        
        // calculate suns declination
        double dec = calcSunDeclination(t);
        return dec;
    }
    
    /**
     * Transposition : calculates solar radiation on tilted surface for date (year, month, day, hour, min, sec)
     * @param lat double : latitude [deg]
     * @param lon double : longitude [deg]
     * @param alt double : altitude [m / mer]
     * @param inc double : inclinaison tilted surface [deg; 0 = horizontal; 90 = vertical]
     * @param paz double : azimut of tilted surface [deg; 0 = north; 90 = east; 180 = south; ...]
     * @param albedo double : ground reflectance albedo
     * @param year double : time
     * @param month double
     * @param day double
     * @param hour double
     * @param min double
     * @param sec double
     * @param zone double : timezone
     * @param dls double : daylight savings ?
     * @return double : global radiation on tilted surface [W/m2]
     */
    public double getTransposition(double lat, double lon, double alt, double paz, double inc, double albedo,
                        double year, double month, double day, double hour, double min, double sec, double zone, double dls,
                                   double[][] horizonData) {

        // calculate solar constant
        double doy = dateToDOY((int)month, (int)day);
        double I0 = 1353 * (1 + 0.033 * cos(doy * 2 * PI / 365.25)); // [W/m]

        // calculate suns declination
        double dec = toRadians(calcSunDeclination(year, month, day, hour, min, sec, zone, dls));
        
        // calculate suns position
        double[] result = calcSunPosition(lat, lon, year, month, day, hour, min, sec, zone, dls, true);
        double az = toRadians(result[0]); // azimuth
        double hc = toRadians(result[1]); // altitude
        double Oz = toRadians(result[2]); // zenit angle
        double ha = getHorizonAngle(result[0], horizonData);

        double mair; // optical mass of air
        double dirh; // horizontal beam radiation
        double difh; // horizontal diffuse radiation
        double gloh; // horizontal global radiation

        double ioh;  // radiation outside atmonsphere
        double hi;   // angle of beam radiation on tilted surface
        double hih;
        double diri; // beam radiation on tilted surface
        double refi; // reflected radiation from ground
        double difi; // diffuse radiation on tilted surface

        if (hc > 0) { // only if sun is above horizon (> 0)
            mair = (1 - 0.1 * alt / 1000) / (sin(hc) + 0.15 * pow((hc + 3.885), -1.253));
            dirh = I0 * sin(hc) * exp(-0.16 - 0.22 * mair);
            difh = 160 * pow((sin(hc)), 0.8) * (1 - 0.17 * (sin(dec) / sin(0.3987)));
            gloh = dirh + difh;
            ioh  = I0 * sin(hc); // irradiation on tilted surface outside atmosphere
            hi = -1 * (acos(cos(toRadians(inc)) * cos(Oz) + sin(toRadians(inc)) * sin(Oz) * cos(az - toRadians(paz))) - PI / 2); // angle of beam radiation on tilted surface (RAD)

            // transposition direct radiation (Hay's model)
            // only when sun is visible and above horizon and obstructions
            if (hi > 0 && toDegrees(hc) > ha) {
                hih = sin(hi) / sin(hc);
                diri = dirh * hih;
            } else {
                hih = 0;
                diri = 0;
                gloh = difh;
            }
            
            // transposition diffuse radiation
            double k = dirh / I0;
            refi = albedo * gloh * 0.5 * (1 - cos(toRadians(inc))); // radiation reflected from ground
            difi = difh * (k * hih + (1 - k) * 0.5 * (1 + cos(toRadians(inc)))) + refi;
            
        } else {
            diri = 0;
            difi = 0;
        }
        
        double Gs = diri + difi; // global radiation on tilted surface
        return Gs;
    }
    
    /**
     * Override method for Transposition : calculates without obstructions (set to zero)
     * @param lat double
     * @param lon double
     * @param alt double
     * @param paz double
     * @param inc double
     * @param albedo double
     * @param year double
     * @param month double
     * @param day double
     * @param hour double
     * @param min double
     * @param sec double
     * @param zone double
     * @param dls double
     * @return double : global solar radiation [W/m2]
     */
    public double getTransposition(double lat, double lon, double alt, double paz, double inc, double albedo,
                        double year, double month, double day, double hour, double min, double sec, double zone, double dls) {
        double[][] horizonData = {{ 0, 180, 360}, {0, 0, 0}}; // horizon dummy
        double Gs = getTransposition(lat, lon, alt, paz, inc, albedo, year, month, day, hour, min, sec, zone, dls, horizonData);
        return Gs;
    }
    
    
    /*
     ---------------------------------------------------------------------------------------
     Calculation of local times of sunrise, solar noon, and sunset
     based on the calculation procedure by NOAA in the javascript in
     http://www.srrb.noaa.gov/highlights/sunrise/sunrise.html and
     http://www.srrb.noaa.gov/highlights/sunrise/azel.html

     Five functions are available:

       - sunrise(lat, lon, year, month, day, timezone, dlstime)
       - solarnoon(lat, lon, year, month, day, timezone, dlstime)
       - sunset(lat, lon, year, month, day, timezone, dlstime)
       - solarazimuth(lat, lon, year, month, day, hour, minute, second, timezone, dlstime)
       - solarelevation(lat, lon, year, month, day, hour, minute, second, timezone, dlstime)

     The sign convention for inputs to the functions named sunrise, solarnoon,
     sunset, solarazimuth, and solarelevationis:

       - positive latitude decimal degrees for northern hemisphere
       - negative longitude degrees for western hemisphere
       - negative time zone hours for western hemisphere

     The other functions use the original
     NOAA sign convention of positive longitude in the western hemisphere.

     The calculations in the NOAA Sunrise/Sunset and Solar Position
     Calculators are based on equations from Astronomical Algorithms,
     by Jean Meeus. NOAA also included atmospheric refraction effects.
     The sunrise and sunset results were reported by NOAA
     to be accurate to within +/- 1 minute for locations between +/- 72
     latitude, and within ten minutes outside of those latitudes.

     This translation was tested for selected locations
     and found to provide results within +/- 1 minute of the
     original Javascript code.

     This translation does not include calculation of prior or next
     susets for locations above the Arctic Circle and below the
     Antarctic Circle, when a sunrise or sunset does not occur.

     Translated from NOAA's Javascript to Excel VBA by:
     Greg Pelletier
     Department of Ecology
     P.O.Box 47600
     Olympia, WA 98504-7600
     email: gpel461@ ecy.wa.gov

    Translated from Greg's VBA to Java by:
    Peter Gallinelli
    the passive igloo project 2015 / 2016
    http//igloo.sailworks.net
    ---------------------------------------------------------------------------------------
    */

    /**
     * Calculate julian day from calendar day
     * @param year int : 4 digit year
     * @param month int : January = 1
     * @param day int : 1 - 31
     * @return double : julian day
     */
    private int calcJD(double year, double month, double day) {
        double A;
        double B;
        double JD;
        if (month <= 2) {
            year = year - 1;
            month = month + 12;
        }
        A = floor(year / 100);
        B = 2 - A + floor(A / 4);
        JD = floor(365.25 * (year + 4716)) + floor(30.6001 * (month + 1)) + day + B - 1524.5;
        return (int) JD;
    }

    /**
     * Convert Julian Day to centuries since J2000.0
     * @param JD int : the Julian Day to convert
     * @return double : value corresponding to the Julian Day
     */
    private double calcTimeJulianCent(double JD) {
        double t = (JD - 2451545) / 36525;
        return t;
    }

    /**
     * Convert centuries since J2000.0 to Julian Day
     * @param t double : number of Julian centuries since J2000.0
     * @return int : the Julian Day corresponding to the t value
     */
    private double calcJDFromJulianCent(double t) {
        double JD = t * 36525 + 2451545;
        return JD;
    }

    /**
     * Calculate the Geometric Mean Longitude of the Sun
     * @param t double : t : number of Julian centuries since J2000.0
     * @return double the Geometric Mean Longitude of the Sun in degrees
     */
    private double calcGeomMeanLongSun(double t) {
        double l0 = 280.46646 + t * (36000.76983 + 0.0003032 * t);
        while ((l0 > 360) || (l0 < 0)) {
            if (l0 > 360)
                l0 = l0 - 360;
            if (l0 < 0)
                l0 = l0 + 360;
        }
        return l0;
    }

    /**
     * calculate the Geometric Mean Anomaly of the Sun
     * @param t double : number of Julian centuries since J2000.0
     * @return double : the Geometric Mean Anomaly of the Sun in degrees
     */
    private double calcGeomMeanAnomalySun(double t) {
        double m = 357.52911 + t * (35999.05029 - 0.0001537 * t);
        return m;
    }

    /**
     * calculate the eccentricity of earth's orbit
     * @param t double : number of Julian centuries since J2000.0
     * @return double : the unitless eccentricity
     */
    private double calcEccentricityEarthOrbit(double t) {
        double e = 0.016708634 - t * (0.000042037 + 0.0000001267 * t);
        return e;
    }

    /**
     * calculate the equation of center for the sun
     * @param t double : number of Julian centuries since J2000.0
     * @return double : in degrees
     */
    private double calcSunEqOfCenter(double t) {
        double m = calcGeomMeanAnomalySun(t);

        double mrad = toRadians(m);
        double sinm = sin(mrad);
        double sin2m = sin(mrad + mrad);
        double sin3m = sin(mrad + mrad + mrad);

        double c =
            sinm * (1.914602 - t * (0.004817 + 0.000014 * t)) + sin2m * (0.019993 - 0.000101 * t) + sin3m * 0.000289;
        return c;
    }

    /**
     * calculate the true longitude of the sun
     * @param t : number of Julian centuries since J2000.0
     * @return O double : sun's true longitude in degrees
     */
    private double calcSunTrueLong(double t) {
        double l0 = calcGeomMeanLongSun(t);
        double c = calcSunEqOfCenter(t);
        double O = l0 + c;
        return O;
    }

    /**
     * calculate the true anamoly of the sun
     * @param t : number of Julian centuries since J2000.0
     * @return double : sun's true anamoly in degrees
     */
    private double calcSunTrueAnomaly(double t) {
        double m = calcGeomMeanAnomalySun(t);
        double c = calcSunEqOfCenter(t);
        double v = m + c;
        return v;
    }

    /**
     * calculate the distance to the sun in AU
     * @param t : number of Julian centuries since J2000.0
     * @return double : sun radius vector in AUs
     */
    private double calcSunRadVector(double t) {
        double v = calcSunTrueAnomaly(t);
        double e = calcEccentricityEarthOrbit(t);
        double R = (1.000001018 * (1 - e * e)) / (1 + e * cos(toRadians(v)));
        return R;
    }

    /**
     * calculate the apparent longitude of the sun
     * @param t : number of Julian centuries since J2000.0
     * @return double : sun's apparent longitude in degrees
     */
    private double calcSunApparentLong(double t) {
        double O = calcSunTrueLong(t);
        double omega = 125.04 - 1934.136 * t;
        double lambda = O - 0.00569 - 0.00478 * sin(toRadians(omega));
        return lambda;
    }

    /**
     * calculate the mean obliquity of the ecliptic
     * @param t : number of Julian centuries since J2000.0
     * @return double : mean obliquity in degrees
     */
    private double calcMeanObliquityOfEcliptic(double t) {
        double seconds = 21.448 - t * (46.815 + t * (0.00059 - t * (0.001813)));
        double e0 = 23 + (26 + (seconds / 60)) / 60;
        return e0;
    }

    /**
     * calculate the corrected obliquity of the ecliptic
     * @param t : number of Julian centuries since J2000.0
     * @return double : corrected obliquity in degrees
     */
    private double calcObliquityCorrection(double t) {
        double e0 = calcMeanObliquityOfEcliptic(t);
        double omega = 125.04 - 1934.136 * t;
        double e = e0 + 0.00256 * cos(toRadians(omega));
        return e;
    }

    /**
     * calculate the right ascension of the sun
     * @param t : number of Julian centuries since J2000.0
     * @return double : sun's right ascension in degrees
     */
    private double calcSunRtAscension(double t) {
        double e = calcObliquityCorrection(t);
        double lambda = calcSunApparentLong(t);

        double tananum = (cos(toRadians(e)) * sin(toRadians(lambda)));
        double tanadenom = (cos(toRadians(lambda)));

        //original NOAA code using javascript Atan2(y,x) convention:
        double alpha = toDegrees(atan2(tananum, tanadenom));
        return alpha;
    }

    /**
     * calculate the declination of the sun
     * @param t : number of Julian centuries since J2000.0
     * @return double : sun's declination in degrees
     */
    private double calcSunDeclination(double t) {
        double e = calcObliquityCorrection(t);
        double lambda = calcSunApparentLong(t);

        double sint = sin(toRadians(e)) * sin(toRadians(lambda));
        double theta = toDegrees(asin(sint));
        return theta;
    }

    /**
     * calculate the difference between true solar time and mean solar time
     * @param t : number of Julian centuries since J2000.0
     * @return double : equation of time in minutes of time
     */
    private double calcEquationOfTime(double t) {
        double epsilon = calcObliquityCorrection(t);
        double l0 = calcGeomMeanLongSun(t);
        double e = calcEccentricityEarthOrbit(t);
        double m = calcGeomMeanAnomalySun(t);
        double y = tan(toRadians(epsilon) / 2);
        y = pow(y, 2);
        double sin2l0 = sin(2 * toRadians(l0));
        double sinm = sin(toRadians(m));
        double cos2l0 = cos(2 * toRadians(l0));
        double sin4l0 = sin(4 * toRadians(l0));
        double sin2m = sin(2 * toRadians(m));

        double Etime =
            y * sin2l0 - 2 * e * sinm + 4 * e * y * sinm * cos2l0 - 0.5 * y * y * sin4l0 - 1.25 * e * e * sin2m;
        return toDegrees(Etime) * 4;
    }

    /**
     * calculate the hour angle of the sun at dawn for the latitude and user selected solar depression below horizon
     * @param lat double : latitude of observer in degrees
     * @param solarDec double : declination angle of sun in degrees
     * @param solardepression double : angle of the sun below the horizion in degrees
     * @return double : hour angle of dawn in radians
     */
    private double calcHourAngleDawn(double lat, double solarDec, double solardepression) {
        double latRad = toRadians(lat);
        double sdRad = toRadians(solarDec);
        /*double HAarg =
            (cos(toRadians(90 + solardepression)) / (cos(latRad) * cos(sdRad)) -
             tan(latRad) * tan(sdRad));*/
        double HA =
            (acos(cos(toRadians(90 + solardepression)) / (cos(latRad) * cos(sdRad)) -
                       tan(latRad) * tan(sdRad)));
        return HA;
    }

    /**
     * calculate the hour angle of the sun at sunrise for the latitude<br>
     * Note: For sunrise and sunset calculations, we assume 0.833 of atmospheric refraction<br>
     * For details about refraction see http://www.srrb.noaa.gov/highlights/sunrise/calcdetails.html
     * @param lat double : latitude of observer in degrees
     * @param solarDec double : declination angle of sun in degrees
     * @return double : hour angle of sunrise in radians
     */
    private double calcHourAngleSunrise(double lat, double solarDec) {
        double latRad = toRadians(lat);
        double sdRad = toRadians(solarDec);

        /*double HAarg =
            (cos(toRadians(90.833)) / (cos(latRad) * cos(sdRad)) - tan(latRad) * tan(sdRad));*/

        double HA =
            (acos(cos(toRadians(90.833)) / (cos(latRad) * cos(sdRad)) -
                       tan(latRad) * tan(sdRad)));
        return HA;
    }

    /**
     * calculate the hour angle of the sun at sunset for the latitude
     * Note: For sunrise and sunset calculations, we assume 0.833 of atmospheric refraction<br>
     * For details about refraction see http://www.srrb.noaa.gov/highlights/sunrise/calcdetails.html
     * @param lat double : latitude of observer in degrees
     * @param solarDec double : declination angle of sun in degrees
     * @return double : hour angle of sunset in radians
     */
    private double calcHourAngleSunset(double lat, double solarDec) {
        double latRad = toRadians(lat);
        double sdRad = toRadians(solarDec);

        /*double HAarg =
            (cos(toRadians(90.833)) / (cos(latRad) * cos(sdRad)) - tan(latRad) * tan(sdRad));*/

        double HA = (acos(cos(toRadians(90.833)) / (cos(latRad) * cos(sdRad)) - tan(latRad) * tan(sdRad)));
        return -HA;
    }

    /**
     * calculate the hour angle of the sun at dusk for the latitude<br>
     * for user selected solar depression below horizon
     * @param lat double : latitude of observer in degrees
     * @param solarDec double : declination angle of sun in degrees
     * @param solardepression double : angle of sun below horizon in degrees
     * @return double : hour angle of dusk in radians
     */
    private double calcHourAngleDusk(double lat, double solarDec, double solardepression) {
        double latRad = toRadians(lat);
        double sdRad = toRadians(solarDec);
        /*double HAarg =
            (cos(degToRad(90 + solardepression)) / (cos(latRad) * cos(sdRad)) -
             tan(latRad) * tan(sdRad));*/
        double HA =
            (acos(cos(toRadians(90 + solardepression)) / (cos(latRad) * cos(sdRad)) -
                       tan(latRad) * tan(sdRad)));
        return -HA;
    }

    /**
     * calculate the Universal Coordinated Time (UTC) of dawn<br>
     * for the given day at the given location on earth<br>
     * for user selected solar depression below horizon
     * @param JD double : julian day
     * @param latitude double : latitude of observer in degrees
     * @param longitude double : longitude of observer in degrees
     * @param solardepression double : angle of sun below the horizon in degrees
     * @return double : time in minutes from zero Z
     */
    private double calcDawnUTC(double JD, double latitude, double longitude, double solardepression) {
        double t = calcTimeJulianCent(JD);

        // *** First pass to approximate sunrise
        double eqtime = calcEquationOfTime(t);
        double solarDec = calcSunDeclination(t);
        double hourangle = calcHourAngleSunrise(latitude, solarDec);
        double delta = longitude - toDegrees(hourangle);
        double timeDiff = 4 * delta;
        // in minutes of time
        double timeUTC = 720 + timeDiff - eqtime;
        // in minutes

        // *** Second pass includes fractional jday in gamma calc
        double newt = calcTimeJulianCent(calcJDFromJulianCent(t) + timeUTC / 1440);
        eqtime = calcEquationOfTime(newt);
        solarDec = calcSunDeclination(newt);
        hourangle = calcHourAngleDawn(latitude, solarDec, solardepression);
        delta = longitude - toDegrees(hourangle);
        timeDiff = 4 * delta;
        timeUTC = 720 + timeDiff - eqtime;
        // in minutes

        return timeUTC;
    }

    /**
     * calculate the Universal Coordinated Time (UTC) of sunrise<br
     * for the given day at the given location on earth
     * @param JD doble : julian day
     * @param latitude double : latitude of observer in degrees
     * @param longitude double : longitude of observer in degrees
     * @return double : time in minutes from zero Z
     */
    private double calcSunriseUTC(double JD, double latitude, double longitude) {
        double t = calcTimeJulianCent(JD);

        // *** First pass to approximate sunrise
        double eqtime = calcEquationOfTime(t);
        double solarDec = calcSunDeclination(t);
        double hourangle = calcHourAngleSunrise(latitude, solarDec);

        double delta = longitude - toDegrees(hourangle);
        double timeDiff = 4 * delta;
        // in minutes of time
        double timeUTC = 720 + timeDiff - eqtime;
        //in minutes

        // *** Second pass includes fractional jday in gamma calc
        double newt = calcTimeJulianCent(calcJDFromJulianCent(t) + timeUTC / 1440);
        eqtime = calcEquationOfTime(newt);
        solarDec = calcSunDeclination(newt);
        hourangle = calcHourAngleSunrise(latitude, solarDec);
        delta = longitude - toDegrees(hourangle);
        timeDiff = 4 * delta;
        timeUTC = 720 + timeDiff - eqtime;
        // in minutes

        return timeUTC;
    }

    /**
     * calculate the Universal Coordinated Time (UTC) of solar<br>
     * noon for the given day at the given location on earth
     * @param t double : number of Julian centuries since J2000.0
     * @param longitude double : longitude of observer in degrees
     * @return double : time in minutes from zero Z
     */
    private double calcSolNoonUTC(double t, double longitude) {
        double newt = calcTimeJulianCent(calcJDFromJulianCent(t) + 0.5 + longitude / 360);

        double eqtime = calcEquationOfTime(newt);
        //double solarNoonDec = calcSunDeclination(newt);
        double solNoonUTC = 720 + (longitude * 4) - eqtime;

        return solNoonUTC;
    }

    /**
     * calculate the Universal Coordinated Time (UTC) of sunset<br>
     * for the given day at the given location on earth
     * @param JD double : julian day
     * @param latitude double : latitude of observer in degrees
     * @param longitude double : longitude of observer in degrees
     * @return double : time in minutes from zero Z
     */
    private double calcSunsetUTC(double JD, double latitude, double longitude) {
        double t = calcTimeJulianCent(JD);

        // First calculates sunrise and approx length of day
        double eqtime = calcEquationOfTime(t);
        double solarDec = calcSunDeclination(t);
        double hourangle = calcHourAngleSunset(latitude, solarDec);
        double delta = longitude - toDegrees(hourangle);
        double timeDiff = 4 * delta;
        double timeUTC = 720 + timeDiff - eqtime;

        // first pass used to include fractional day in gamma calc
        double newt = calcTimeJulianCent(calcJDFromJulianCent(t) + timeUTC / 1440);
        eqtime = calcEquationOfTime(newt);
        solarDec = calcSunDeclination(newt);
        hourangle = calcHourAngleSunset(latitude, solarDec);
        delta = longitude - toDegrees(hourangle);
        timeDiff = 4 * delta;
        timeUTC = 720 + timeDiff - eqtime;
        // in minutes

        return timeUTC;
    }

    /**
     * calculate the Universal Coordinated Time (UTC) of dusk<br>
     * for the given day at the given location on earth<br>
     * for user selected solar depression below horizon
     * @param JD double : julian day
     * @param latitude double : latitude of observer in degrees
     * @param longitude double : longitude of observer in degrees
     * @param solardepression double : angle of sun below horizon
     * @return double : time in minutes from zero Z
     */
    private double calcDuskUTC(double JD, double latitude, double longitude, double solardepression) {
        double t = calcTimeJulianCent(JD);

        // First calculates sunrise and approx length of day
        double eqtime = calcEquationOfTime(t);
        double solarDec = calcSunDeclination(t);
        double hourangle = calcHourAngleSunset(latitude, solarDec);
        double delta = longitude - toDegrees(hourangle);
        double timeDiff = 4 * delta;
        double timeUTC = 720 + timeDiff - eqtime;

        // first pass used to include fractional day in gamma calc
        double newt = calcTimeJulianCent(calcJDFromJulianCent(t) + timeUTC / 1440);
        eqtime = calcEquationOfTime(newt);
        solarDec = calcSunDeclination(newt);
        hourangle = calcHourAngleDusk(latitude, solarDec, solardepression);
        delta = longitude - toDegrees(hourangle);
        timeDiff = 4 * delta;
        timeUTC = 720 + timeDiff - eqtime;
        // in minutes

        return timeUTC;
    }

    /**
     * calculate time of dawn  for the entered date and location
     * For latitudes greater than 72 degrees N and S, calculations are<br>
     * accurate to within 10 minutes. For latitudes less than +/- 72<br>
     * accuracy is approximately one minute.<br>
     * NOTE: longitude is negative for West<br>
     *    functions convert longitude to positive for the western hemisphere for calls to<br>
     *    other functions using the original sign convention from the NOAA javascript code.
     * @param lat double : latitude (decimal degrees)
     * @param lon double : longitude (decimal degrees)
     * @param year double : year
     * @param month double : month
     * @param day double : day
     * @param timezone double : time zone hours relative to GMT/UTC (hours)
     * @param dlstime double : daylight savings time (0 = no, 1 = yes) (hours)
     * @param solardepression double : angle of sun below horizon in degrees
     * @return double : dawn time in local time (days)
     */
    public double calcDawn(double lat, double lon, double year, double month, double day, double timezone,
                           double dlstime, double solardepression) {
        // change sign convention for longitude from negative to positive in western hemisphere
        double longitude = lon * -1;
        double latitude = lat;
        if (latitude > 89.8)
            latitude = 89.8;
        if (latitude < -89.8)
            latitude = -89.8;

        double JD = calcJD(year, month, day);

        // Calculate sunrise for this date
        double riseTimeGMT = calcDawnUTC(JD, latitude, longitude, solardepression);

        //  adjust for time zone and daylight savings time in minutes
        double riseTimeLST = riseTimeGMT + (60 * timezone) + (dlstime * 60);

        //  convert to days
        return riseTimeLST / 1440;
    }

    /**
     * calculate time of sunrise  for the entered date and location.
     * For latitudes greater than 72 degrees N and S, calculations are<br>
     * accurate to within 10 minutes. For latitudes less than +/- 72<br>
     * accuracy is approximately one minute.<br>
     * NOTE: longitude is negative for West<br>
     *    functions convert longitude to positive for the western hemisphere for calls to<br>
     *    other functions using the original sign convention from the NOAA javascript code.
     * @param lat double : latitude (decimal degrees)
     * @param lon double : longitude (decimal degrees)
     * @param year double : year
     * @param month double : month
     * @param day double : day
     * @param timezone double : time zone hours relative to GMT/UTC (hours)
     * @param dlstime double : daylight savings time (0 = no, 1 = yes) (hours)
     * @return double : sunrise time in local time (days)
     */
    public double calcSunrise(double lat, double lon, double year, double month, double day, double timezone,
                              double dlstime) {
        // change sign convention for longitude from negative to positive in western hemisphere
        double longitude = lon * -1;
        double latitude = lat;
        if (latitude > 89.8)
            latitude = 89.8;
        if (latitude < -89.8)
            latitude = -89.8;

        double JD = calcJD(year, month, day);

        // Calculate sunrise for this date
        double riseTimeGMT = calcSunriseUTC(JD, latitude, longitude);

        //  adjust for time zone and daylight savings time in minutes
        double VriseTimeLST = riseTimeGMT + (60 * timezone) + (dlstime * 60);

        //  convert to days
        return VriseTimeLST / 1440;
    }

    /**
     * calculate the Universal Coordinated Time (UTC) of solar noon for the given day at the given location on earth
     * For latitudes greater than 72 degrees N and S, calculations are<br>
     * accurate to within 10 minutes. For latitudes less than +/- 72<br>
     * accuracy is approximately one minute.<br>
     * NOTE: longitude is negative for West<br>
     *    functions convert longitude to positive for the western hemisphere for calls to<br>
     *    other functions using the original sign convention from the NOAA javascript code.
     * @param lat double : latitude (decimal degrees)
     * @param lon double : longitude (decimal degrees)
     * @param year double : year
     * @param month double : month
     * @param day double : day
     * @param timezone double : time zone hours relative to GMT/UTC (hours)
     * @param dlstime double : daylight savings time (0 = no, 1 = yes) (hours)
     * @return double : solar noon in local time
     */
    public double calcSolarNoon(double lat, double lon, double year, double month, double day, double timezone,
                                double dlstime) {
        // change sign convention for longitude from negative to positive in western hemisphere
        double longitude = lon * -1;
        double latitude = lat;
        if (latitude > 89.8)
            latitude = 89.8;
        if (latitude < -89.8)
            latitude = -89.8;

        double JD = calcJD(year, month, day);
        double t = calcTimeJulianCent(JD);

        double newt = calcTimeJulianCent(calcJDFromJulianCent(t) + 0.5 + longitude / 360);

        double eqtime = calcEquationOfTime(newt);
        //double solarNoonDec = calcSunDeclination(newt);
        double solNoonUTC = 720 + (longitude * 4) - eqtime;

        //  adjust for time zone and daylight savings time in minutes
        double solarnoon = solNoonUTC + (60 * timezone) + (dlstime * 60);

        //  convert to days
        return solarnoon / 1440;
    }

    /**
     * calculate time of sunrise and sunset for the entered date and location.
     * For latitudes greater than 72 degrees N and S, calculations are<br>
     * accurate to within 10 minutes. For latitudes less than +/- 72<br>
     * accuracy is approximately one minute.<br>
     * NOTE: longitude is negative for West<br>
     *    functions convert longitude to positive for the western hemisphere for calls to<br>
     *    other functions using the original sign convention from the NOAA javascript code.
     * @param lat double : latitude (decimal degrees)
     * @param lon double : longitude (decimal degrees)
     * @param year double : year
     * @param month double : month
     * @param day double : day
     * @param timezone double : time zone hours relative to GMT/UTC (hours)
     * @param dlstime double : daylight savings time (0 = no, 1 = yes) (hours)
     * @return double : sunrise time in local time (days)
     */
    public double calcSunset(double lat, double lon, double year, double month, double day, double timezone,
                             double dlstime) {
        // change sign convention for longitude from negative to positive in western hemisphere
        double longitude = lon * -1;
        double latitude = lat;
        if (latitude > 89.8)
            latitude = 89.8;
        if (latitude < -89.8)
            latitude = -89.8;

        double JD = calcJD(year, month, day);

        // Calculate sunset for this date
        double setTimeGMT = calcSunsetUTC(JD, latitude, longitude);

        //  adjust for time zone and daylight savings time in minutes
        double setTimeLST = setTimeGMT + (60 * timezone) + (dlstime * 60);

        //  convert to days
        return setTimeLST / 1440;
    }

    /**
     * calculate time of sunrise and sunset for the entered date and location.
     * For latitudes greater than 72 degrees N and S, calculations are<br>
     * accurate to within 10 minutes. For latitudes less than +/- 72<br>
     * accuracy is approximately one minute.<br>
     * NOTE: longitude is negative for West<br>
     *    functions convert longitude to positive for the western hemisphere for calls to<br>
     *    other functions using the original sign convention from the NOAA javascript code.
     * @param lat double : latitude (decimal degrees)
     * @param lon double : longitude (decimal degrees)
     * @param year double : year
     * @param month double : month
     * @param day double : day
     * @param timezone double : time zone hours relative to GMT/UTC (hours)
     * @param dlstime double : daylight savings time (0 = no, 1 = yes) (hours)
     * @return double : sunrise time in local time (days)
     */
    public double calcDusk(double lat, double lon, double year, double month, double day, double timezone,
                           double dlstime, double solardepression) {
        // change sign convention for longitude from negative to positive in western hemisphere
        double longitude = lon * -1;
        double latitude = lat;
        if (latitude > 89.8)
            latitude = 89.8;
        if (latitude < -89.8)
            latitude = -89.8;

        double JD = calcJD(year, month, day);

        // Calculate sunset for this date
        double setTimeGMT = calcDuskUTC(JD, latitude, longitude, solardepression);

        //  adjust for time zone and daylight savings time in minutes
        double setTimeLST = setTimeGMT + (60 * timezone) + (dlstime * 60);

        //  convert to days
        return setTimeLST / 1440;
    }

    /**
     * calculate solar azimut (deg from north) and altitude (deg from horizontal for the given date, time and location.<br>
     * NOTE: longitude is negative for West<br>
     *    functions convert longitude to positive for the western hemisphere for calls to<br>
     *    other functions using the original sign convention from the NOAA javascript code.
     * @param lat double : latitude (decimal degrees)
     * @param lon double : longitude (decimal degrees)
     * @param year double : date year
     * @param month double : date month (January = 1)
     * @param day double : date day (1 to 31)
     * @param hours double : time hour (0 to 23)
     * @param minutes double : time minute (0 to 59)
     * @param seconds double : time second (0 to 59)
     * @param timezone double : time zone hours relative to GMT/UTC (hours)
     * @param dlstime double : daylight savings time (0 = no, 1 = yes) (hours)
     * @param refractionOn boolean : true = YES, false = NO
     * @return double[] : [0] solar azimut in degrees from north [1] solar elevation in degrees from horizon [2] zenith angle
     */
    public double[] calcSunPosition(double lat, double lon, double year, double month, double day, double hours,
                                    double minutes, double seconds, double timezone, double dlstime,
                                    boolean refractionOn) {
        // change sign convention for longitude from negative to positive in western hemisphere
        double longitude = lon * -1;
        double latitude = lat;
        if (latitude > 89.99)
            latitude = 89.99;
        if (latitude < -89.99)
            latitude = -89.99;

        // change time zone to positive hours in western hemisphere
        double zone = timezone * -1;
        double daySavings = dlstime * 60;
        double hh = hours - (daySavings / 60);
        double mm = minutes;
        double ss = seconds;

        // timenow is GMT time for calculation in hours since 0Z
        double timenow = hh + mm / 60 + ss / 3600 + zone;

        double JD = calcJD(year, month, day);
        double t = calcTimeJulianCent(JD + timenow / 24);
        double R = calcSunRadVector(t);
        //double Valpha = calcSunRtAscension(t);
        double theta = calcSunDeclination(t);
        double Etime = calcEquationOfTime(t);

        double eqtime = Etime;
        double solarDec = theta; // in degrees
        //double earthRadVec = R;

        double solarTimeFix = eqtime - 4 * longitude + 60 * zone;
        double trueSolarTime = hh * 60 + mm + ss / 60 + solarTimeFix;
        // in minutes

        while (trueSolarTime > 1440)
            trueSolarTime = trueSolarTime - 1440;

        double hourangle = trueSolarTime / 4 - 180;
        if (hourangle < -180)
            hourangle = hourangle + 360;

        double harad = toRadians(hourangle);
        double csZ =
            sin(toRadians(latitude)) * sin(toRadians(solarDec)) +
            cos(toRadians(latitude)) * cos(toRadians(solarDec)) * cos(harad);

        if (csZ > 1)
            csZ = 1;
        else if (csZ < -1)
            csZ = -1;

        double zenith = toDegrees(acos(csZ));
        double azDenom = (cos(toRadians(latitude)) * sin(toRadians(zenith)));
        double azimuth = 0;
        if (abs(azDenom) > 0.001) {
            double azRad =
                ((sin(toRadians(latitude)) * cos(toRadians(zenith))) - sin(toRadians(solarDec))) / azDenom;
            if (abs(azRad) > 1) {
                if (azRad < 0)
                    azRad = -1;
                else
                    azRad = 1;
            }
            azimuth = 180 - toDegrees(acos(azRad));
            if (hourangle > 0)
                azimuth = -azimuth;
        } else {
            if (latitude > 0)
                azimuth = 180;
            else
                azimuth = 0;
        }
        if (azimuth < 0)
            azimuth = azimuth + 360;

        // atmospheric refraction correction
        double exoatmElevation = 90 - zenith;
        double refractionCorrection = 0;
        if (refractionOn) {
            if (exoatmElevation > 85)
                refractionCorrection = 0;
            else {
                double te = tan(toRadians(exoatmElevation));
                if (exoatmElevation > 5)
                    refractionCorrection = 58.1 / te - 0.07 / (te * te * te) + 0.000086 / (te * te * te * te * te);
                else if (exoatmElevation > -0.575) {
                    double step1 = (-12.79 + exoatmElevation * 0.711);
                    double step2 = (103.4 + exoatmElevation * (step1));
                    double step3 = (-518.2 + exoatmElevation * (step2));
                    refractionCorrection = 1735 + exoatmElevation * (step3);
                } else
                    refractionCorrection = -20.774 / te;

                refractionCorrection = refractionCorrection / 3600;
            }
        }
        double solarzen = zenith - refractionCorrection;
        double solarazimuth = azimuth;
        double solarelevation = 90 - solarzen;
        double[] result = { solarazimuth, solarelevation, solarzen};
        return result;
    }
}
